// Cart Management System
class CartManager {
    constructor() {
        this.cart = this.loadCart();
        this.apiUrl = window.location.origin;
        this.updateCartDisplay();
    }

    loadCart() {
        const saved = localStorage.getItem('ai_commerce_cart');
        return saved ? JSON.parse(saved) : [];
    }

    saveCart() {
        localStorage.setItem('ai_commerce_cart', JSON.stringify(this.cart));
        this.updateCartDisplay();
    }

    addToCart(serviceId, serviceName, price) {
        const existingItem = this.cart.find(item => item.serviceId === serviceId);
        
        if (existingItem) {
            existingItem.quantity += 1;
        } else {
            this.cart.push({
                serviceId: serviceId,
                serviceName: serviceName,
                price: parseFloat(price),
                quantity: 1
            });
        }
        
        this.saveCart();
        this.showCartNotification(`${serviceName} added to cart!`);
    }

    removeFromCart(serviceId) {
        this.cart = this.cart.filter(item => item.serviceId !== serviceId);
        this.saveCart();
    }

    updateQuantity(serviceId, quantity) {
        const item = this.cart.find(item => item.serviceId === serviceId);
        if (item) {
            if (quantity <= 0) {
                this.removeFromCart(serviceId);
            } else {
                item.quantity = quantity;
                this.saveCart();
            }
        }
    }

    getTotal() {
        return this.cart.reduce((total, item) => total + (item.price * item.quantity), 0);
    }

    getCartCount() {
        return this.cart.reduce((count, item) => count + item.quantity, 0);
    }

    clearCart() {
        this.cart = [];
        this.saveCart();
    }

    updateCartDisplay() {
        const cartCountElements = document.querySelectorAll('.cart-count');
        const count = this.getCartCount();
        
        cartCountElements.forEach(element => {
            element.textContent = count;
            element.style.display = count > 0 ? 'inline' : 'none';
        });
    }

    showCartNotification(message) {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = 'cart-notification';
        notification.textContent = message;
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: #00ff88;
            color: #0a0a0a;
            padding: 12px 20px;
            border-radius: 8px;
            font-weight: 600;
            z-index: 10000;
            transform: translateX(400px);
            transition: transform 0.3s ease;
        `;
        
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.style.transform = 'translateX(0)';
        }, 100);
        
        // Remove after 3 seconds
        setTimeout(() => {
            notification.style.transform = 'translateX(400px)';
            setTimeout(() => {
                document.body.removeChild(notification);
            }, 300);
        }, 3000);
    }

    buyNow(serviceId, serviceName, price) {
        // Clear cart and add single item
        this.cart = [{
            serviceId: serviceId,
            serviceName: serviceName,
            price: parseFloat(price),
            quantity: 1
        }];
        this.saveCart();
        
        // Redirect to checkout
        window.location.href = '../checkout.html';
    }
}

// Initialize cart manager
const cartManager = new CartManager();

// Add cart icon to header if it doesn't exist
document.addEventListener('DOMContentLoaded', function() {
    // Add cart icon to navigation
    const nav = document.querySelector('nav');
    if (nav && !document.querySelector('.cart-icon')) {
        const cartIcon = document.createElement('div');
        cartIcon.className = 'cart-icon';
        cartIcon.innerHTML = `
            <a href="/cart.html" style="color: #00ff88; text-decoration: none; position: relative;">
                <i class="fas fa-shopping-cart" style="font-size: 18px;"></i>
                <span class="cart-count" style="
                    position: absolute;
                    top: -8px;
                    right: -8px;
                    background: #ff6b35;
                    color: white;
                    border-radius: 50%;
                    width: 20px;
                    height: 20px;
                    font-size: 12px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    font-weight: bold;
                ">0</span>
            </a>
        `;
        nav.appendChild(cartIcon);
    }
    
    cartManager.updateCartDisplay();
});